using System;
using System.Globalization;
using System.Text.RegularExpressions;

namespace Precisely_single
{
    /// <summary>
    /// A class for rounding off numbers in standard decimal notation.
    /// </summary>
    public static class Precise
    {
        /// <summary>
        /// Determines if the specified string represents a number in standard
        /// decimal notation.
        /// </summary>
        /// <param name="number">
        /// The string to be tested.
        /// </param>
        /// <returns>
        /// A value indicating whether <paramref name="number"/> is a string
        /// representing a number in standard decimal notation.
        /// </returns>
        /// <remarks>
        /// A string representing a number in standard decimal notation must
        /// contain at least one decimal digit, it may also contain the decimal
        /// separator <c>.</c> and have a leading sign <c>+</c> or <c>-</c>.
        /// No other characters are allowed.
        /// Such a string may still be unconvertible to a floating point number
        /// if it represents a value which is too large or too small.
        /// This function doesn't throw an exception if the specified string is
        /// a null reference; instead, <c>false</c> is returned.
        /// </remarks>
        public static bool IsStdDecimal(string number)
        {
            return
                !Object.ReferenceEquals(number, null)
                &&
                Regex.IsMatch(number, @"^[+-]?(?:\d+(?:\.\d*)?|\.?\d+)$");
        }

        /// <summary>
        /// Rounds off a number in standard decimal notation by reducing the
        /// number of digits after the decimal separator in a way that the value
        /// obtained by converting the number into single precision floating
        /// point format is not affected.
        /// This is possible because of the limited precision of that data type.
        /// </summary>
        /// <param name="number">
        /// A string containing the number to be rounded off.
        /// </param>
        /// <param name="fiveDown">
        /// Specifies whether a number whose last digit is 5 should be rounded
        /// down when the last digit is truncated.
        /// </param>
        /// <returns>
        /// A string containing the number rounded off.
        /// </returns>
        /// <exception cref="System.ArgumentNullException">
        /// <paramref name="number"/> is a null reference.
        /// </exception>
        /// <exception cref="System.ArgumentException">
        /// <paramref name="number"/> does not represent a number in standard
        /// decimal notation.
        /// </exception>
        /// <exception cref="System.OverflowException">
        /// The value of <paramref name="number"/> exceeds the range of single
        /// precision floating point values.
        /// </exception>
        /// <remarks>
        /// This function does not remove unnecessary digits before the decimal
        /// separator.
        /// </remarks>
        /// <seealso cref="IsStdDecimal"/>
        public static string RoundOff(string number, bool fiveDown)
        {
            if (!IsStdDecimal(number))
                throw
                    !Object.ReferenceEquals(number, null) ?
                    new ArgumentException() :
                    new ArgumentNullException();
            NumberFormatInfo nfi = CultureInfo.InvariantCulture.NumberFormat;
            // The next line may throw an OverflowException.
            float oldValue = float.Parse(number, nfi);
            if (number.IndexOf('.') < 0) return number;
            Regex noDigitRegex = new Regex(@"^[+-]?$");
            bool pointFound = false;
            for (; ; )
            {
                int lastIndex = number.Length - 1;
                int lastChar = number[lastIndex];
                string oldNumber = number;
                number = number.Remove(lastIndex);
                if (lastChar == '.')
                {
                    if (noDigitRegex.IsMatch(number)) number += '0';
                    return number;
                }
                bool increase = lastChar > '5' || lastChar == '5' && !fiveDown;
                if (!increase)
                    fiveDown = false;
                else
                {
                    while ((lastChar = number[--lastIndex]) == '9') ;
                    if (lastChar == '.')
                    {
                        int pointIndex = lastIndex;
                        while (
                            --lastIndex >= 0 &&
                            (lastChar = number[lastIndex]) == '9') ;
                        number = number.Remove(lastIndex + 1);
                        number =
                            (!noDigitRegex.IsMatch(number) ?
                                number.Remove(lastIndex) +
                                (char)(lastChar + 1) :
                                number + '1') +
                                new string('0', pointIndex - (lastIndex + 1));
                        pointFound = true;
                    }
                    else
                    {
                        number =
                            number.Remove(lastIndex) + (char)(lastChar + 1);
                        fiveDown = lastChar == '4';
                    }
                }
                float newValue;
                try
                {
                    // OverflowException may happen here if the absolute value
                    // of newValue is not less than
                    // 340282356779733642999999999999999999999.5 and less than
                    // 340282356779733643000000000000000000000.
                    newValue = float.Parse(number, nfi);
                }
                // FormatException is thrown if newNumber is ".", "+." or "-.".
                catch (FormatException)
                {
                    newValue = 0;
                }
                if (oldValue != newValue) return oldNumber;
                if (pointFound) return number;
            }
        }
    }
}
